<?php

/*
 * this class should be used to stores properties and methods shared by the
 * admin and public side of wordpress
 */
class Daln_Shared
{

    //regex
    public $hex_rgb_regex = '/^#(?:[0-9a-fA-F]{3}){1,2}$/';
    public $font_family_regex = '/^([A-Za-z0-9-\'", ]*)$/';
    public $url_regex = '/^(http|https):\/\/[-A-Za-z0-9+&@#\/%?=~_|$!:,.;]+$/';
    public $regex_capability = '/^\s*[A-Za-z0-9_]+\s*$/';

    protected static $instance = null;

    private $data = array();

    private function __construct()
    {

        //Set plugin textdomain
        load_plugin_textdomain('daln', false, 'live-news/lang/');
        
        $this->data['slug'] = 'daln';
        $this->data['ver'] = '2.14';
        $this->data['dir'] = substr(plugin_dir_path(__FILE__), 0, -7);
        $this->data['url'] = substr(plugin_dir_url(__FILE__), 0, -7);

    }

    public static function get_instance()
    {

        if (null == self::$instance) {
            self::$instance = new self;
        }

        return self::$instance;

    }

    //retrieve data
    public function get($index)
    {
        return $this->data[$index];
    }

    /*
     * Convert a numeric target to a textual target
     *
     * @param $target_id int
     * @return string
     */
    public function get_textual_target($target_id){

        switch($target_id){

            case 1:

                return 'Website';
                break;

            case 2:

                return 'URL';
                break;

        }

    }

    /*
     * Convert a numeric source to a textual source
     *
     * @param $source_id int
     * @return string
     */
    public function get_textual_source($source_id){

        switch($source_id){

            case 1:

                return __('Manually Added', 'daln');
                break;

            case 2:

                return __('Posts', 'daln');
                break;

            case 3:

	            return __('RSS', 'daln');
	            break;

            case 4:

	            return __('Twitter', 'daln');
	            break;

        }

    }

    /*
     * Retrieve the ticker name from the ticker id
     *
     * @param $ticker_id int
     * @return string
     */
    public function get_textual_ticker($ticker_id){

        global $wpdb;
        $table_name = $wpdb->prefix . $this->get('slug') . "_tickers";
        $safe_sql = $wpdb->prepare("SELECT name FROM $table_name WHERE id = %d ", $ticker_id);
        $ticker_obj = $wpdb->get_row($safe_sql);

        if($ticker_obj !== NULL){
            return $ticker_obj->name;
        }else{
            return 'Invalid Ticker ID';
        }

    }

    /*
     * Generate a short version of a string without truncating words
     *
     * @param $str The string
     * @param $length The maximum length of the string
     * @return string The short version of the string
     */
    public function strlen_no_truncate( $str, $length ){

        if (mb_strlen($str) > $length){
            $str = wordwrap($str, $length);
            $str = mb_substr($str, 0, mb_strpos($str, "\n"));
            $str = $str . ' ...';
        }

        return $str;

    }

    /*
     * Returns true if the ticker is used in sliding news or in featured news
     *
     * @param $ticker_id int
     * @return bool True if the ticker is used or False if the ticker is not used
     */
    public function ticker_is_used($ticker_id){

        global $wpdb;

        //verify if the ticker is used in the featured news
        $table_name = $wpdb->prefix . $this->get('slug') . "_featured_news";
        $safe_sql = $wpdb->prepare("SELECT COUNT(*) FROM $table_name WHERE ticker_id = %d", $ticker_id);
        $number_of_uses = $wpdb->get_var($safe_sql);
        if($number_of_uses > 0){return true;}

        //verify if the ticker is used in the sliding news
        $table_name = $wpdb->prefix . $this->get('slug') . "_sliding_news";
        $safe_sql = $wpdb->prepare("SELECT COUNT(*) FROM $table_name WHERE ticker_id = %d", $ticker_id);
        $number_of_uses = $wpdb->get_var($safe_sql);
        if($number_of_uses > 0){return true;}

        return false;

    }

    /*
     * Given a ticker id returns true if the ticker exists or false if the ticker doesn't exist
     *
     * @param $ticker_id int
     * @return bool True if the ticker exists or False if the ticker doesn't exists
     */
    public function ticker_exists($ticker_id){

        global $wpdb;

        $table_name = $wpdb->prefix . $this->get('slug') . "_tickers";
        $safe_sql = $wpdb->prepare("SELECT * FROM $table_name WHERE id = %d", $ticker_id);
        $ticker_obj = $wpdb->get_row($safe_sql);
        if($ticker_obj !== NULL){
            return true;
        }else{
            return false;
        }

    }

    /*
     * Given an hexadecimal rgb color an array with the 3 components converted in decimal is returned
     *
     * @param string The hexadecimal rgb color
     * @return array An array with the 3 component of the color converted in decimal
     */
    public function rgb_hex_to_dec($hex)
    {

        if (mb_strlen($hex) == 3) {
            $r = hexdec(substr($hex, 0, 1) . substr($hex, 0, 1));
            $g = hexdec(substr($hex, 1, 1) . substr($hex, 1, 1));
            $b = hexdec(substr($hex, 2, 1) . substr($hex, 2, 1));
        } else {
            $r = hexdec(substr($hex, 0, 2));
            $g = hexdec(substr($hex, 2, 2));
            $b = hexdec(substr($hex, 4, 2));
        }

        return array(
            'r' => $r,
            'g' => $g,
            'b' => $b
        );

    }

    /*
     * Get the number of tickers
     *
     * @return int The number of tickers
     */
    public function get_number_of_tickers(){

        global $wpdb;
        $table_name = $wpdb->prefix . $this->get('slug') . "_tickers";
        $number_of_tickers = $wpdb->get_var("SELECT COUNT(*) FROM $table_name");

        return $number_of_tickers;

    }

    /*
     * Get the number of featured news
     *
     * @return int The number of featured news
     */
    public function get_number_of_featured_news(){

        global $wpdb;
        $table_name = $wpdb->prefix . $this->get('slug') . "_featured_news";
        $number_of_featured_news = $wpdb->get_var("SELECT COUNT(*) FROM $table_name");

        return $number_of_featured_news;

    }

    /*
     * Get the number of sliding news
     *
     * @return int The number of sliding news
     */
    public function get_number_of_sliding_news(){

        global $wpdb;
        $table_name = $wpdb->prefix . $this->get('slug') . "_sliding_news";
        $number_of_sliding_news = $wpdb->get_var("SELECT COUNT(*) FROM $table_name");

        return $number_of_sliding_news;

    }

    /*
     * Based on the selected option this method is used to remove the following elements included in a tweet:
     *
     * - links
     * - hashtags
     * - usernames
     *
     * @param $tweet string The tweet
     * @param $strip_links bool Whether to strip links
     * @param $strip_hashtags bool Whether to strip hashtags
     * @param $strip_usernames bool Whether to strip usernames
     *
     * @return string The tweet with the elements removed
     */
    public function twitter_remove_elements($tweet, $strip_links, $strip_hashtags, $strip_usernames){

        //strip links (generic url regex which allows almost any url)
        if($strip_links){
            $tweet = preg_replace('/(https?|ftp|file):\/\/[^\s]+/', '', $tweet );
        }

        //strip hashtags
        if($strip_hashtags){
            $tweet = preg_replace('/#[a-zA-Z_]{1}[a-zA-Z0-9_]*/', '', $tweet );
        }

        //strip usernames
	    if($strip_usernames){
		    $tweet = preg_replace('/@[a-zA-Z0-9_]{1,15}/', '', $tweet );
	    }

        return $tweet;

    }

    /*
     * Get the current URL with the method specified with the "Detect URL Mode" option
     */
    public function get_current_url(){

        if(get_option($this->get('slug') . '_detect_url_mode') === 'server_variable'){

            //Detect the URL using the "Server Variable" method
            return is_ssl() ? 'https' : 'http' . '://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];

        }else{

            //Detect the URL using the "WP Request" method
            global $wp;
            return trailingslashit(home_url(add_query_arg(array(),$wp->request)));

        }

    }

    /**
     * Returns the object of the first ticker with the target equal to 2 (url) that can be displayed with the current
     * url.
     *
     * Note that to determine if a ticker can be displayed with the current url the following fields are considered:
     *
     * - Target
     * - Target URL
     * - Target URL Mode
     *
     * @param $current_url The url that should be searched in the target url field of the tickers
     * @return mixed The object with the data of the news ticker of false.
     */
    public function get_ticker_with_target_url($current_url){

        $found = false;
        $ticker_id = null;
        $ticker_obj = false;

        global $wpdb;
        $table_name = $wpdb->prefix . $this->get('slug') . "_tickers";
        $safe_sql = "SELECT * FROM $table_name WHERE target = 2 ORDER BY id ASC";
        $ticker_a = $wpdb->get_results($safe_sql, ARRAY_A);

        foreach($ticker_a as $key => $ticker){

            $url_a = preg_split('/\r\n|[\r\n]/', $ticker['url']);

            if(intval($ticker['url_mode'], 10) === 0){

                //Include

                //Get the ticker_id of the first news ticker that includes the current url
                if($ticker_id !== null){
                    break;
                }

                foreach($url_a as $key2 => $url){
                    if($url === $current_url){
                        $found = true;
                    }
                }

                if($found){
                    $ticker_id = $ticker['id'];
                    break;
                }

            }else{

                //Exclude

                //Get the ticker_id of the first news ticker that doesn't include the current url
                foreach($url_a as $key2 => $url){
                    if($url === $current_url){
                        $found = true;
                    }
                }

                if(!$found){
                    $ticker_id = $ticker['id'];
                    break;
                }

            }

        }

        //Get the object of the news ticker that includes the current url
        if($ticker_id !== NULL){
            global $wpdb;
            $table_name = $wpdb->prefix . $this->get('slug') . "_tickers";
            $safe_sql = $wpdb->prepare("SELECT * FROM $table_name WHERE id = %d", $ticker_id);
            $ticker_obj = $wpdb->get_row($safe_sql);
        }

        return $ticker_obj;

    }

}